package gov.va.vamf.scheduling.direct;

import gov.va.vamf.scheduling.direct.datalayer.eligibility.DirectBookingEligibilityCriteriaService;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.RequestEligibilityCriteriaService;
import gov.va.vamf.scheduling.direct.datalayer.facility.CustomFriendlyTextService;
import gov.va.vamf.scheduling.direct.datalayer.facility.InstitutionsDAOImpl;
import gov.va.vamf.scheduling.direct.domain.AppointmentEligibilityAtInstitution;
import gov.va.vamf.scheduling.direct.domain.CoreSetting;
import gov.va.vamf.scheduling.direct.domain.CoreSettings;
import gov.va.vamf.scheduling.direct.domain.CustomFriendlyText;
import gov.va.vamf.scheduling.direct.domain.CustomRequestSetting;
import gov.va.vamf.scheduling.direct.domain.Institution;
import gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria;
import gov.va.vamf.scheduling.direct.eligibilitychecker.ExpressCareTimes;
import gov.va.vamf.scheduling.direct.eligibilitychecker.PatientClinicalServicesEligibilityChecker;
import gov.va.vamf.scheduling.facility.timezone.FacilityTimezone;
import gov.va.vamf.scheduling.facility.timezone.FacilityTimezoneService;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static gov.va.vamf.scheduling.direct.domain.CoreSettings.DEFAULT_TIMEZONE;

@Service
public class InstitutionService {
    @Resource
    private DirectBookingEligibilityCriteriaService directBookingEligibilityCriteriaService;

    @Resource
    private RequestEligibilityCriteriaService requestEligibilityCriteriaService;

    @Resource
    private FacilityTimezoneService facilityTimezoneService;

    @Resource
    private PatientClinicalServicesEligibilityChecker patientClinicalServicesEligibilityChecker;

    @Resource
    InstitutionsDAOImpl institutionsDao;

    @Resource
    CustomFriendlyTextService customFriendlyTextService;

    public Collection<AppointmentEligibilityAtInstitution> getInstitutionsEligibleForClinicalService(
            final Collection<String> facilitySiteCodes,
            final String parentCode,
            final String clinicalServiceId) {

        final List<AppointmentEligibilityAtInstitution> institutionEligibilityList;

        List<Institution> institutions = institutionsDao.getInstitutions(facilitySiteCodes, parentCode);

        if (clinicalServiceId.equals(CoreSettings.EXPRESS_CARE_ID)) {
            institutionEligibilityList = getInstitutionsEligibleForExpressCare(institutions);
        }
        else {
            institutionEligibilityList = new ArrayList<>(institutions.size());

            final List<String> institutionCodes = new ArrayList<>(institutions.size());

            for (final Institution institution : institutions) {
                institutionCodes.add(institution.getInstitutionCode());
            }

            final Map<String, CoreSetting> directBookingSettingsMap =
                    directBookingEligibilityCriteriaService.fetchSettingAsMap(institutionCodes, clinicalServiceId);

            final Map<String, CoreSetting> requestSettingsMap =
                    requestEligibilityCriteriaService.fetchSettingAsMap(institutionCodes, clinicalServiceId);

            for (final Institution institution : institutions) {
                boolean directSchedulingSupported = false;

                final CoreSetting directBookingSettings =
                        directBookingSettingsMap.get(institution.getInstitutionCode());

                if (directBookingSettings != null) {
                    directSchedulingSupported = directBookingSettings.isEnabled();
                }

                boolean requestSupported = false;

                final CoreSetting requestSettings =
                        requestSettingsMap.get(institution.getInstitutionCode());

                if (requestSettings != null) {
                    requestSupported = requestSettings.isEnabled();
                }

                if (directSchedulingSupported || requestSupported) {
                    FacilityTimezone facilityTimezone = facilityTimezoneService.fetchFacilityTimezoneById(institution.getRootStationCode());
                    String institutionTimezone = (facilityTimezone != null && facilityTimezone.getTimezone() != null) ? facilityTimezone.getTimezone() : DEFAULT_TIMEZONE;

                    final AppointmentEligibilityAtInstitution appointmentEligibility = new AppointmentEligibilityAtInstitution();
                    setCustomFriendlyName(institution);
                    appointmentEligibility.setInstitution(institution);
                    appointmentEligibility.setDirectSchedulingSupported(directSchedulingSupported);
                    appointmentEligibility.setRequestSupported(requestSupported);
                    appointmentEligibility.setInstitutionTimezone(institutionTimezone);
                    institutionEligibilityList.add(appointmentEligibility);
                }
            }
        }

        return institutionEligibilityList;
    }

    /**
     * Get institutions eligible for express care that are supported now.
     * Also sets express care end text.
     *
     * @param institutions
     */
    public List<AppointmentEligibilityAtInstitution> getInstitutionsEligibleForExpressCare(final List<Institution> institutions) {
        final List<AppointmentEligibilityAtInstitution> institutionEligibilityList = new ArrayList<>(institutions.size());

        for (final Institution institution : institutions) {
            ExpressCareTimes times = null;

            final RequestEligibilityCriteria criteria =
                    requestEligibilityCriteriaService.retrieveRequestEligibilityCriteria(institution.getInstitutionCode());

            if (criteria != null && criteria.getCustomRequestSettings() != null) {
                for (final CustomRequestSetting setting : criteria.getCustomRequestSettings()) {
                    if (setting.getId().equalsIgnoreCase(CoreSettings.EXPRESS_CARE_ID) && setting.getSupported() != null && setting.getSupported()) {
                        times = patientClinicalServicesEligibilityChecker.getExpressCareTimes(institution.getRootStationCode(), setting);
                        break;
                    }
                }
            }

            // filter on express care supported institution
            if (times != null) {
                final AppointmentEligibilityAtInstitution eligibility = new AppointmentEligibilityAtInstitution();
                setCustomFriendlyName(institution);
                eligibility.setInstitution(institution);
                eligibility.setDirectSchedulingSupported(false);
                eligibility.setRequestSupported(true);
                eligibility.setExpressTimes(times);
                institutionEligibilityList.add(eligibility);
            }
        }

        return institutionEligibilityList;
    }

    public Collection<Institution> getParentInstitutions(List<String> rootFacilityCodes) {
        return institutionsDao.getParentInstitutions(rootFacilityCodes);
    }

    public void setCustomFriendlyName(Institution institution) {
        CustomFriendlyText customFriendlyText = customFriendlyTextService.fetchByInstitutionCode(institution.getInstitutionCode());

        if (customFriendlyText != null && !customFriendlyText.getFriendlyText().isEmpty()) {
            institution.setName(customFriendlyText.getFriendlyText());
        }
        else {
            institution.setName(institution.getAuthoritativeName());
        }
    }

    public Institution getInstitution(final String institutionCode) {
        return institutionsDao.getInstitution(institutionCode);
    }
}
